%-----------------------------------
%	LIFE-CYCLE
%`	CLOSED ECONOMY
%	WITH NO GOVERNMENT
%   REPLICATES THE MODEL IN GERTLER (1999)
%   VERSION BY LUKASZ RACHEL
%-----------------------------------

%-----------------------------------
%	THIS VERSION: 16 SEP 2018
%-----------------------------------

close all;

%-----------------------------------
%	ENDOGENOUS VARIABLES 1+2+5+1+6+4+3=22
%-----------------------------------

var 

% demographics (1)
psi 

% interest rates, prices (2)
R w

% marginal propensities to consume and discounting adjustments (5)
eps xiw xir bigom Gamma

% distribution of wealth (1)
lam 

% aggregate quantities (6)
c ass h inv y k

% group quantities (4)
cr cw assr assw

% government (lump sum tax, social security wealth of retirees and of workers) (3)
tax s sw;
 
%-----------------------------------
%	EXOGENOUS VARIABLES
%-----------------------------------

varexo 

% TFP
x

% demographics 
n ggamma oomega 

% government
bbar gbar ebar hbar;

%-----------------------------------
%	PARAMETERS
%-----------------------------------

parameters

% technology and preferences
aalpha bbeta ddelta ssigma 

% steady state life expectancy
ggamma_ss n_ss;

%-----------------------------------
%	CALIBRATION
%-----------------------------------

%@# include "demog_rr_param.mod"
% these replicate Gertler 1999 model as much as possible:

%aalpha=0.667; %labor share
%bbeta=0.98;   %discount factor
%ddelta=0.1;   %depreciation
%ssigma=0.5;   %IES
%oomega_ss=0.977; %probability of remaining a worker
%x_ss=1.01;       %TFP growth (gross)
%ggamma_ss=0.9;%Probability of dying as a retiree
%n_ss=1.01;    %Population growth (gross)

load my_parameters.mat 

set_param_value('aalpha',aalpha)
set_param_value('bbeta',bbeta)
set_param_value('ddelta',ddelta)
set_param_value('ssigma',ssigma)
set_param_value('ggamma_ss',ggamma_ss)
set_param_value('n_ss',n_ss)

%------------------------------------
%	MODEL
%------------------------------------

model;

% 1) Dependency ratio
n * psi = (1 - oomega) + ggamma * psi(-1);

% 2) Retirees MPC
xir = 1 - ggamma * (bbeta ^ ssigma) * (R ^ (ssigma - 1)) * xir / xir(+1);

% 3) Workers MPC
xiw = 1 - (bbeta ^ ssigma) * ((bigom(+1) * R) ^ (ssigma - 1)) * xiw / xiw(+1);

% 4) Ratio of MPC
eps = xir / xiw;

% 5) Adjustment factor
bigom = oomega + (1 - oomega) * (eps ^ (1 / (1 - ssigma)));

% 6) Aggregate consumption
c =  xiw*((1-lam(-1))*R(-1)*ass(-1)+h+sw + eps*(lam(-1)*R(-1)*ass(-1)+s));

% 7) Workers consumption
cw = xiw * (R(-1) * assw(-1) + h + sw);

% 8) Retirees consumption
cr = xir * (R(-1) * assr(-1) + s);

% 9) Human wealth
h = aalpha*y - tax + x * oomega * h(+1) / (bigom(+1) * R);

% 10) Distribution of wealth 
x * n * (lam - 1 + oomega) * ass / oomega 
= (1 - xir) * lam(-1) * R(-1) * ass(-1) + (ebar+hbar)*y - xir*s;

% 12) Real wage
w = aalpha * y;

% 13) Rental rate
R = ((1 - aalpha) * (y / k(-1)))+(1-ddelta);

% 14) Production function
y = k(-1) ^ (1 - aalpha);

% 15) Aggregate assets
ass = k + bbar*y;

% 16) Workers assets
assw = (1 - lam(-1)) * ass;

% 17) Retirees assets
assr = lam(-1) * ass;

% 18) Capital accumulation
x * n(+1) * k = (1 - ddelta) * k(-1) + inv;

% 21) Resource constraint
y = c + inv + gbar*y;

% 22) Gamma: the human capital difference due to higher discounting
Gamma = (1-((n*x)/R))/(1-(x*oomega)/(R*bigom));

% 23) Government budget constraint
bbar(+1)*y(+1)*x*n(+1) = R(-1)*bbar*y + gbar*y + (ebar+hbar)*y - tax;

% 24) Retiree's social security wealth
s = (ebar+hbar)*y + (x*s(+1)*ggamma)/R;

% 25) Workers' social security wealth
sw = (1-oomega)*(eps(+1)*s(+1)*x/(R*bigom(+1)*psi)) + sw(+1)*x*oomega/(R*bigom(+1));

end;

%--------------------------------------
%	INITIAL STEADY STATE
%--------------------------------------

% NOTE: Gertler (1999) Table 1 reports k and h; these are K/Y ratios;
% Below, k and h denote K/XN and H/XN

initval;

x     = x_init_ss;
n     = n_init_ss;
oomega= oomega_init_ss;
ggamma= ggamma_init_ss;

gbar  = gbar_init_ss;
bbar  = bbar_init_ss;
ebar  = ebar_init_ss;
hbar  = hbar_init_ss;
psi   = (1 - oomega) / (n - ggamma);
R	  = 1.049;
k	  = 3.391;
inv	  = (x * n + ddelta - 1) * k;
y	  = k ^ (1 - aalpha);
ass   = k + bbar*y;
w     = aalpha * y;
bigom = 1.039;
xiw	  = 1-((R*bigom)^(ssigma-1))*bbeta^ssigma;
xir   = 1-(R^(ssigma-1))*(bbeta^ssigma)*ggamma;
eps	  = xir / xiw;
lam   = 0.177;
h	  = 6.9237;
Gamma = (1-(n*x)/R)/(1-(x)/(R*bigom));
s     = (1/(1-ggamma*(1+x)/R))*(ebar+hbar);
sw    = Gamma*((1-oomega)*eps)/(bigom*R)*(s/psi*(x))/(1-(x*n)/R);
c     = xiw*((1-lam)*R*ass+h+sw+eps*(lam*R*ass+s));

end;
steady(solve_algo=0);
check;

%--------------------------------------
%	PERMANENT SHOCK - ENDVAL:
%--------------------------------------









simul(periods=1000);
